/*
 * @(#)SimulatedTable.java  1.0  2006-01-18
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.dml.model;

import ch.randelshofer.util.*;
import ch.hslu.cm.simulation.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.util.*;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
/**
 * A simulated table.
 *
 * @author  Werner Randelshofer
 * @version 1.0 30. Januar 2004  Created.
 */
public class SimulatedTable extends AbstractElement 
        implements TableModel {
    private String tableName;
    private String[] columnNames;
    private Class[] columnClasses;
    private Object[][] tableData;
    
    /** Creates a new instance. */
    public SimulatedTable() {
        tableName = DMLModel.labels.getString("defaultTableName");
        
        columnNames = new String[5];
        for (int i=0; i < columnNames.length; i++) {
            columnNames[i] = DMLModel.labels.getFormatted("defaultColumnName", new Object[] {i});
            columnClasses[i] = String.class;
        }
        columnNames = new String[5];
        tableData = new Object[5][5];
    }
    
    public void setName(String name) {
        if (! this.tableName.equals(name)) {
            this.tableName = name;
            fireTableChanged();
        }
    }
    public String getName() {
        return tableName;
    }
    
    public void addSimulatedTableListener(SimulatedTableListener l) {
        listenerList.add(SimulatedTableListener.class, l);
    }
    
    public void removeSimulatedTableListener(SimulatedTableListener l) {
        listenerList.remove(SimulatedTableListener.class, l);
    }
    
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireTableChanged() {
        SimulatedTableEvent event = null;
        
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==SimulatedTableListener.class) {
                // Lazily create the event:
                if (event == null)
                    event = new SimulatedTableEvent(this);
                ((SimulatedTableListener)listeners[i+1]).tableChanged(event);
            }
        }
    }
    
    public SimulatedTable clone() {
        SimulatedTable that = (SimulatedTable) super.clone();
        return that;
    }
    
    public void write(DOMOutput out) {
        out.addAttribute("name", tableName);
    }
    public void read(DOMInput in) {
        tableName = in.getAttribute("name", DMLModel.labels.getString("defaultTableName"));
    }
    public DMLModel getDMLModel() {
        return (DMLModel) getSimulation();
    }
    
    public void addedToSimulation(SimulatedObjectEvent e) {
    }
    
    public void connectionAdded(SimulatedObjectEvent e) {
    }
    
    public void connectionRemoved(SimulatedObjectEvent e) {
    }
    
    public void elementChanged(SimulatedObjectEvent e) {
    }
    
    public void elementRequestRemove(SimulatedObjectEvent e) {
    }
    
    public void removedFromSimulation(SimulatedObjectEvent e) {
    }

    /**
     * Removes a listener from the list that is notified each time a
     * change to the data model occurs.
     * 
     * 
     * @param l		the TableModelListener
     */
    public void removeTableModelListener(TableModelListener l) {
        listenerList.remove(TableModelListener.class, l);
    }

    /**
     * Adds a listener to the list that is notified each time a change
     * to the data model occurs.
     * 
     * 
     * @param l		the TableModelListener
     */
    public void addTableModelListener(TableModelListener l) {
        listenerList.add(TableModelListener.class, l);
    }

    /**
     * Sets the value in the cell at <code>columnIndex</code> and
     * <code>rowIndex</code> to <code>aValue</code>.
     * 
     * 
     * @param aValue		 the new value
     * @param rowIndex	 the row whose value is to be changed
     * @param columnIndex 	 the column whose value is to be changed
     * @see #getValueAt
     * @see #isCellEditable
     */
    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        tableData[rowIndex][columnIndex] = aValue;
        fireTableCellUpdated(rowIndex, columnIndex);
    }

    /**
     * Returns the tableName of the column at <code>columnIndex</code>.  This is used
     * to initialize the table's column header tableName.  Note: this tableName does
     * not need to be unique; two columns in a table can have the same tableName.
     * 
     * 
     * 
     * @param columnIndex	the index of the column
     * @return the tableName of the column
     */
    public String getColumnName(int columnIndex) {
        return columnNames[columnIndex];
    }

    /**
     * Returns the most specific superclass for all the cell values 
     * in the column.  This is used by the <code>JTable</code> to set up a 
     * default renderer and editor for the column.
     * 
     * 
     * @param columnIndex  the index of the column
     * @return the common ancestor class of the object values in the model.
     */
    public Class<?> getColumnClass(int columnIndex) {
        return columnClasses[columnIndex];
    }

    /**
     * Returns true if the cell at <code>rowIndex</code> and
     * <code>columnIndex</code>
     * is editable.  Otherwise, <code>setValueAt</code> on the cell will not
     * change the value of that cell.
     * 
     * 
     * @param rowIndex	the row whose value to be queried
     * @param columnIndex	the column whose value to be queried
     * @return true if the cell is editable
     * @see #setValueAt
     */
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return true;
    }

    /**
     * Returns the value for the cell at <code>columnIndex</code> and
     * <code>rowIndex</code>.
     * 
     * 
     * @param rowIndex	the row whose value is to be queried
     * @param columnIndex 	the column whose value is to be queried
     * @return the value Object at the specified cell
     */
    public Object getValueAt(int rowIndex, int columnIndex) {
        return tableData[rowIndex][columnIndex];
    }

    /**
     * Returns an ID specifying the concept simulated by this element.
     */
    public int getSimulatedConcept() {
        return DMLModel.TABLE;
    }

    /**
     * Returns the number of rows in the model. A
     * <code>JTable</code> uses this method to determine how many rows it
     * should display.  This method should be quick, as it
     * is called frequently during rendering.
     * 
     * 
     * @return the number of rows in the model
     * @see #getColumnCount
     */
    public int getRowCount() {
        return tableData.length;
    }

    /**
     * Returns the number of columns in the model. A
     * <code>JTable</code> uses this method to determine how many columns it
     * should create and display by default.
     * 
     * 
     * @return the number of columns in the model
     * @see #getRowCount
     */
    public int getColumnCount() {
        return columnNames.length;
    }
    
//
//  Fire methods
//

    /**
     * Notifies all listeners that all cell values in the table's
     * rows may have changed. The number of rows may also have changed
     * and the <code>JTable</code> should redraw the
     * table from scratch. The structure of the table (as in the order of the
     * columns) is assumed to be the same.
     *
     * @see TableModelEvent
     * @see EventListenerList
     * @see javax.swing.JTable#tableChanged(TableModelEvent)
     */
    public void fireTableDataChanged() {
        fireTableChanged(new TableModelEvent(this));
    }

    /**
     * Notifies all listeners that the table's structure has changed.
     * The number of columns in the table, and the names and types of
     * the new columns may be different from the previous state.
     * If the <code>JTable</code> receives this event and its
     * <code>autoCreateColumnsFromModel</code>
     * flag is set it discards any table columns that it had and reallocates
     * default columns in the order they appear in the model. This is the
     * same as calling <code>setModel(TableModel)</code> on the
     * <code>JTable</code>.
     *
     * @see TableModelEvent
     * @see EventListenerList
     */
    public void fireTableStructureChanged() {
        fireTableChanged(new TableModelEvent(this, TableModelEvent.HEADER_ROW));
    }

    /**
     * Notifies all listeners that rows in the range
     * <code>[firstRow, lastRow]</code>, inclusive, have been inserted.
     *
     * @param  firstRow  the first row
     * @param  lastRow   the last row
     *
     * @see TableModelEvent
     * @see EventListenerList
     *
     */
    public void fireTableRowsInserted(int firstRow, int lastRow) {
        fireTableChanged(new TableModelEvent(this, firstRow, lastRow,
                             TableModelEvent.ALL_COLUMNS, TableModelEvent.INSERT));
    }

    /**
     * Notifies all listeners that rows in the range
     * <code>[firstRow, lastRow]</code>, inclusive, have been updated.
     *
     * @param firstRow  the first row
     * @param lastRow   the last row
     *
     * @see TableModelEvent
     * @see EventListenerList
     */
    public void fireTableRowsUpdated(int firstRow, int lastRow) {
        fireTableChanged(new TableModelEvent(this, firstRow, lastRow,
                             TableModelEvent.ALL_COLUMNS, TableModelEvent.UPDATE));
    }

    /**
     * Notifies all listeners that rows in the range
     * <code>[firstRow, lastRow]</code>, inclusive, have been deleted.
     *
     * @param firstRow  the first row
     * @param lastRow   the last row
     *
     * @see TableModelEvent
     * @see EventListenerList
     */
    public void fireTableRowsDeleted(int firstRow, int lastRow) {
        fireTableChanged(new TableModelEvent(this, firstRow, lastRow,
                             TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
    }

    /**
     * Notifies all listeners that the value of the cell at 
     * <code>[row, column]</code> has been updated.
     *
     * @param row  row of cell which has been updated
     * @param column  column of cell which has been updated
     * @see TableModelEvent
     * @see EventListenerList
     */
    public void fireTableCellUpdated(int row, int column) {
        fireTableChanged(new TableModelEvent(this, row, row, column));
    }

    /**
     * Forwards the given notification event to all
     * <code>TableModelListeners</code> that registered
     * themselves as listeners for this table model.
     *
     * @param e  the event to be forwarded
     *
     * @see #addTableModelListener
     * @see TableModelEvent
     * @see EventListenerList
     */
    public void fireTableChanged(TableModelEvent e) {
	// Guaranteed to return a non-null array
	Object[] listeners = listenerList.getListenerList();
	// Process the listeners last to first, notifying
	// those that are interested in this event
	for (int i = listeners.length-2; i>=0; i-=2) {
	    if (listeners[i]==TableModelListener.class) {
		((TableModelListener)listeners[i+1]).tableChanged(e);
	    }
	}
    }
}
